--[[---------------------------------------------------------------------------
	Chocolatier Two Laboratory UI
	Copyright (c) 2006-2007 Big Splash Games, LLC. All Rights Reserved.
--]]---------------------------------------------------------------------------

gLabDiscovery = nil

local self = gDialogTable.lab
local char = gDialogTable.char
local slotCount = gDialogTable.limit
local type = gDialogTable.type
local possibleTypes = gDialogTable.possibleTypes or {}
local possibleRecipes = gDialogTable.possibleRecipes or {}
local possibleCount = gDialogTable.possibleCount or 0

local width=650
local height=480
local windowX = 75 + ui.xOffset
local charX = width-ui.charW

-------------------------------------------------------------------------------

local delicious = "characters/taster_delicious"
local needs_something = "characters/taster_needs_something"
local other = "characters/taster_asconce"
local neutral = "characters/ba_labkeep"
local no_good = "characters/taster_no_good"
local yuck = "characters/taster_yuck"

-------------------------------------------------------------------------------
-- History display

local function UpdateHistory()
	for i,h in ipairs(self.history) do
		for j=1,6 do
			local ing = h[j][1]
			local good = h[j][2]
			local ingBitmap = "history_"..tostring(i).."_"..tostring(j)
			local badBitmap = "bad_"..tostring(i).."_"..tostring(j)
			
			if ing == "" then
				EnableWindow(ingBitmap, false)
				EnableWindow(badBitmap, false)
			else
				EnableWindow(ingBitmap, true)
				SetBitmap(ingBitmap, "item/"..ing)
				EnableWindow(badBitmap, not good)
			end
		end
	end
	if table.getn(self.history) < 5 then
		for i=table.getn(self.history)+1,5 do
			for j=1,6 do
				local ingBitmap = "history_"..tostring(i).."_"..tostring(j)
				local badBitmap = "bad_"..tostring(i).."_"..tostring(j)
				EnableWindow(ingBitmap, false)
				EnableWindow(badBitmap, false)
			end
		end
	end
end

-------------------------------------------------------------------------------
-- Slot management

local slots = {}
local ingredients = {}
local filled = {}

local function ClearSlots()
	for i=1,6 do
		slots[i] = ""
		SetBitmap("slot_"..tostring(i), "")
	end
	ingredients = {}
	filled = {}
	EnableWindow("check", false)
end

local function SetSlot(i, name)
	local item = LItem:ByName(name)
	slots[i] = name
	filled[i] = item
	SetBitmap("slot_"..tostring(i), "item/"..name)
end

local function FindSlot()
	for i=1,slotCount do
		if not filled[i] then return i end
	end
	return nil
end

local function ClearSlot(i)
	if filled[i] then
		PlaySound("lab_deselect")
		ingredients[slots[i]] = ingredients[slots[i]] - 1
		slots[i] = ""
		filled[i] = null
		SetBitmap("slot_"..tostring(i), "")
		if FindSlot() then
			EnableWindow("check", false)
			SetLabel("reaction",GetString("lab_instructions"))
		end
	end
end

local function AddIngredient(name)
	local n = FindSlot()
	if n then
		PlaySound("sfx/lab_select.ogg")
		SetSlot(n,name)
		
		if not ingredients[name] then ingredients[name] = 1
		else ingredients[name] = ingredients[name] + 1
		end

		SetBitmap("taster", neutral)
		if not FindSlot() then
			SetLabel("reaction",GetString("lab_taste"))
			EnableWindow("check", true)
		end
	end
end

-------------------------------------------------------------------------------
-- Recipe search

local function FindClosestRecipe()
	local t = {}

	-- Find the target recipe closest to the given ingredients
	local targetItem = nil
	local targetProximity = -1
	
	for item in LItem:AllProducts() do
		-- Ignore anything with the wrong number of slots
		if item.recipeSize == slotCount then
			local total = 0
			local correct = 0
			local available = true
			for ing,count in pairs(item.recipe) do
				-- Make sure player has the ingredients to make this recipe before aiming for it
				local item = LItem:ByName(ing)
				if not self.stock[ing] then available = false end
				
				total = total + count
				if ingredients[ing] then
					-- How many of the supplied ingredients will actually be used?
					if ingredients[ing] < count then count = ingredients[ing] end
					correct = correct + count
				end
			end
			
			-- TODO: Give some points for varietal cacao mismatch ?
--			for ing,count in pairs(item.recipe) do
--				if ing.type == "it_cacao" then
--				end
--			end
			
			-- Proximity 0..100, score based on proximity and other factors
			local proximity = bsgFloor(100 * correct / total)
			local score = proximity
			if score == 100 then score = 999999		-- force to the top
			elseif not available then score = 0
			else
				if item and not item.known then score = score + 500 end
				if item and item.lab then score = score + 1000 end
				if item and item.type == type then score = score + 10000 end
			end

			table.insert(t, {proximity=proximity, item=item, score=score})
		end
	end
	
	table.sort(t, function(a,b) return a.score > b.score end)
	
--for _,t in ipairs(t) do
--	DebugOut(tostring(t.score)..":"..GetString(t.item.name))
--end	
	
	if table.getn(t) > 0 then
		return t[1].item, t[1].proximity
	else
		return nil, 0
	end
end

local function Submit()
	local item,proximity = FindClosestRecipe()

	-- Base facial reaction on proximity
	local reaction = yuck
	if proximity > 50 then
		reaction = needs_something
	elseif proximity > 20 then
		reaction = no_good
	end
	
	-- Count cacao varieties as one for basic "need some kind of cacao" feedback
	local cacao = 0
	cacao = cacao + (ingredients.cacao or 0)
	cacao = cacao + (ingredients.ma_cacao or 0)
	cacao = cacao + (ingredients.sj_cacao or 0)
	cacao = cacao + (ingredients.ca_cacao or 0)
	cacao = cacao + (ingredients.ab_cacao or 0)
	cacao = cacao + (ingredients.ja_cacao or 0)
	cacao = cacao + (ingredients.ph_cacao or 0)
	cacao = cacao + (ingredients.am_cacao or 0)
	
	-- Look for specific mismatched ingredients, including varietal cacao
	local cacao_mismatch = false
	local cacao_match = false
	local secret = false
	local fruit = false
	local nut = false
	local flavor = false
	for name,_ in pairs(item.recipe) do
		local ing = LItem:ByName(name)
		if ing.type == "it_cacao" then
			if not ingredients[name] then cacao_mismatch = true	-- AT LEAST ONE VARIETAL MISMATCH
			else cacao_match = true								-- AT LEAST ONE VARIETAL MATCH
			end
		elseif not ingredients[name] then
			if ing.type == "it_secret" then secret = true		-- MISSING SECRET
			elseif ing.type == "it_fruit" then fruit = true		-- MISSING FRUIT
			elseif ing.type == "it_nut" then nut = true			-- MISSING NUT
			elseif ing.type == "it_flavor" then flavor = true	-- MISSING FLAVOR
			end
		end
	end
	
--DebugOut("cacao_mismatch:"..tostring(cacao_mismatch))
--DebugOut("cacao_match:"..tostring(cacao_match))
--DebugOut("secret:"..tostring(secret))
--DebugOut("fruit:"..tostring(fruit))
--DebugOut("nut:"..tostring(nut))
--DebugOut("flavor:"..tostring(flavor))

	-- Determine result/hint string based on specifics
	local result = nil
	
	-- RULE: RECIPE IS ALREADY KNOWN
	if proximity == 100 and item.known then
		reaction = other
		result = GetString("lab_already", GetString(item.name))
		
	-- RULE: RECIPE IS NOT LAB-DISCOVERABLE
	elseif proximity == 100 and not item.lab then
		reaction = needs_something
		result = GetString("lab_nolab")
		
	-- RULE: RECIPE IS CORRECT
	elseif proximity == 100 and item.type == type then
		gSim.inventCount = gSim.inventCount+ 1
		LQuest:IncrementVariable("InventedRecipes")
		LQuest:IncrementVariable("RecentlyInventedRecipes")
		
		reaction = delicious
		result = GetString("lab_discovery", GetString(item.name))
		item:EnableRecipe("lab_recipe")
		gLabDiscovery = item
		self.history = {}
		
		-- Lab not available for some time
		self.available = gSim.weeks + 2

	-- RULE: TARGET IS A DIFFERENT TYPE (not sure this still applies but keeping it anyway)
	elseif item.type ~= type then
		reaction = other
		if item.type.knownCount > 0 then
			result = GetString("lab_differenttype", GetString(type.name), GetString(item.type.name))
		else
			result = GetString("lab_unknowntype", GetString(type.name))
		end

	-- RULE: INCLUDE AT LEAST ONE CACAO OF ANY VARIETY
	elseif cacao == 0 then
		reaction = other
		result = GetString("lab_cacao")

	-- RULE: AT LEAST ONE VARIETAL MATCH, BUT ALSO ONE VARIETAL MISMATCH
	elseif cacao_mismatch and cacao_match then
		result = GetString("lab_blend")

	-- RULE: NO VARIETAL MATCH
	elseif not cacao_match then
		result = GetString("lab_varietal")
		
	-- RULE: AT LEAST ONE SUGAR
	elseif not ingredients.sugar then
		reaction = other
		result = GetString("lab_sugar")
		
	-- RULE: AT LEAST ONE TRUFFLE POWDER WHEN MAKING TRUFFLES
	elseif type.name == "truffle" and (not ingredients.powder) then
		reaction = needs_something
		result = GetString("lab_powder")
		
	-- RULE: MISSED SECRET INGREDIENT
	elseif secret then
		result = GetString("lab_secret")

	-- RULE: MISSED FRUIT
	elseif fruit then
		result = GetString("lab_fruit")

	-- RULE: MISSED NUT
	elseif nut then
		result = GetString("lab_nut")

	-- RULE: MISSED FLAVOR/SPICE
	elseif flavor then
		result = GetString("lab_flavor")
	
	-- RULE: GENERIC PROXIMATE SPECIFIC HINT FOR MORE OF SOMETHING, OR GENERIC "SOMETHING ELSE"
	elseif proximity > 50 then
		reaction = needs_something
		result = GetString("lab_different")
		for ing,count in pairs(item.recipe) do
			if ingredients[ing] and ingredients[ing] < count then
				result = GetString("lab_more", GetString(ing))
				break
			end
		end
	
	-- RULE: GENERIC "NO GOOD"
	elseif proximity > 20 then
		result = GetString("lab_no_good")
	
	-- RULE: BAIL OUT "YUCK!"	
	else
		result = GetString("lab_yuck")
	end
	
	if fpWrite then
		if gLabDiscovery then
			-- FIRSTPEEK: LabSuccess, time-stamp, type, slotsUsed, slot1, slot2, slot3, slot4, slot5, slot6, recipe
			fpWrite { "LabSuccess",type.name,slotCount,tostring(slots[1]),tostring(slots[2]),tostring(slots[3]),tostring(slots[4]),tostring(slots[5]),tostring(slots[6]), item.name }
		else
			-- FIRSTPEEK: LabAttempt, time-stamp, type, slotsUsed, slot1, slot2, slot3, slot4, slot5, slot6
			fpWrite { "LabAttempt",type.name,slotCount,tostring(slots[1]),tostring(slots[2]),tostring(slots[3]),tostring(slots[4]),tostring(slots[5]),tostring(slots[6]) }
		end
	end
	
	if not gLabDiscovery then
		-- Figure out what went wrong (if anything)
		-- which supplied ingredients are not in the recipe,
		-- or which ingredients are used too much?
		local history = {}
--		for ing,_ in pairs(ingredients) do ingredients[ing] = 0 end
		ingredients = {}
		newIngredients = {}
		for i=1,6 do
			local ing = slots[i]
			if ing == "" then
				history[i] = { ing, false }
			else
				ingredients[ing] = (ingredients[ing] or 0) + 1
				if not item then
					history[i] = { ing, false }
				elseif item.known then
					-- Target recipe is known - claer it
					history[i] = { ing, true }
					ClearSlot(i)
				elseif item.recipe[ing] and item.recipe[ing] >= ingredients[ing] then
					-- This one is usable in the target recipe
					history[i] = { ing, true }
					newIngredients[ing] = ingredients[ing]
				else
					-- This one is not usable in the target recipe
					history[i] = { ing, false }
					ClearSlot(i)
				end
			end
		end
		table.insert(self.history, history)
		if table.getn(self.history) > 5 then table.remove(self.history,1) end
		ingredients = newIngredients
	end

	SetLabel("reaction", result)
	SetBitmap("taster", reaction)
	if reaction == delicious then PlaySound("lab_reaction_5")
	elseif reaction == needs_something then PlaySound("lab_reaction_4")
	elseif reaction == neutral then PlaySound("lab_reaction_3")
	elseif reaction == no_good then PlaySound("lab_reaction_2")
	elseif reaction == yuck then PlaySound("lab_reaction_1")
	end
	
--	ClearSlots()
	UpdateHistory()
	
	if gLabDiscovery then
		PopModal("lab")
		DisplayDialog { "ui/lab_discovery.lua" }
	end
end

-------------------------------------------------------------------------------
-- Lab closure

local function CloseLab()
	CloseWindow(gLabDiscovery)
end

-------------------------------------------------------------------------------
-- Experiment type

local typeCounts = { square=3, infusion=4, sauce=4, praline=5, truffle=6, exotic=5 }

local function SetType(t)
	if type.name ~= t then
		SetButtonToggleState("set_"..t, true)
	
		local inventable = false
		local available = false
		-- Are there any inventable recipes left of this type?
		for p in LItem:AllProducts() do
			if p.type.name == t then
				if p.lab and not p.known then inventable = true end
				if possibleRecipes[p.name] then available = true end
			end
		end
		
		if not inventable then
			SetBitmap("taster", neutral)
			SetLabel("reaction", GetString("lab_finishedtype", GetString(t)))
		elseif not available then
			SetBitmap("taster", neutral)
			SetLabel("reaction", GetString("lab_notype", GetString(t), GetString(type.name)))
		else
			PopModal("lab")
			DisplayDialog { "ui/lab_game.lua", lab=self, char=char, possibleTypes=possibleTypes, possibleRecipes=possibleRecipes, possibleCount=possibleCount,
				limit = typeCounts[t],
				type = LProductType:ByName(t) }
		end
	end
end

local y = ui.charH-40
slotCountButtons =
{
--	BeginGroup(),
	
	-- square
	Button { x=332,y=295, name="set_square", type=kToggle,
		graphics= { "", "item/highlight_square" },
		command = function() SetType("square") end,
		Bitmap { x=2,y=2, image="item/b_01" } },

	-- infusion
	Button { x=332,y=325, name="set_infusion", type=kToggle,
		graphics= { "", "item/highlight_infusion" },
		command = function() SetType("infusion") end,
		Bitmap { x=2,y=2, image="item/i_01" } },

	-- sauce
	Button { x=332,y=355, name="set_sauce", type=kToggle,
		graphics= { "", "item/highlight_sauce" },
		command = function() SetType("sauce") end,
		Bitmap { x=2,y=2, image="item/s_01" } },

	-- praline
	Button { x=332,y=385, name="set_praline", type=kToggle,
		graphics= { "", "item/highlight_praline" },
		command = function() SetType("praline") end,
		Bitmap { x=2,y=2, image="item/p_01" } },

	-- truffle
	Button { x=332,y=415, name="set_truffle", type=kToggle,
		graphics= { "", "item/highlight_truffle" },
		command = function() SetType("truffle") end,
		Bitmap { x=2,y=2, image="item/t_01" } },

	-- exotic
	Button { x=332,y=445, name="set_exotic", type=kToggle,
		graphics= { "", "item/highlight_exotic" },
		command = function() SetType("exotic") end,
		Bitmap { x=2,y=2, image="item/e_01" } },
		
	Text { x=367,y=295,w=100,h=36, name="lbl_square", label=LabelString("square"), flags=kVAlignCenter+kHAlignLeft },
	Text { x=367,y=325,w=100,h=36, name="lbl_infusion", label=LabelString("infusion"), flags=kVAlignCenter+kHAlignLeft },
	Text { x=367,y=355,w=100,h=36, name="lbl_sauce", label=LabelString("sauce"), flags=kVAlignCenter+kHAlignLeft },
	Text { x=367,y=385,w=100,h=36, name="lbl_praline", label=LabelString("praline"), flags=kVAlignCenter+kHAlignLeft },
	Text { x=367,y=415,w=100,h=36, name="lbl_truffle", label=LabelString("truffle"), flags=kVAlignCenter+kHAlignLeft },
	Text { x=367,y=445,w=100,h=36, name="lbl_exotic", label=LabelString("exotic"), flags=kVAlignCenter+kHAlignLeft },
}


-------------------------------------------------------------------------------
-- UI construction

function LabItemRollover(name)
	local text = LabelString(name) .. "<br>" .. GetString("click_add")
	local y = 2 * bsgFontHeight(popupFont)
	y = (2*ui.itemHeight - y)/2
	return MakeRollover
	{
		x=0,y=0, color=PopupColor, inset=2, alpha=0.75,
		Bitmap { x=0,y=0,w=2*ui.itemWidth,h=2*ui.itemHeight, image="item/"..name.."_big" },
		TightText { x=2*ui.itemWidth+2,y=y,h=2*ui.itemHeight, label=text, font=popupFont, flags=kHAlignLeft+kVAlignTop },
	}
end

function LabSlotRollover(n)
	if filled[n] then
		local text = LabelString(filled[n].name) .. "<br>" .. GetString("click_remove")
		local y = 2 * bsgFontHeight(popupFont)
		y = (2*ui.itemHeight - y)/2
		return MakeRollover
		{
			x=0,y=0, color=PopupColor, inset=2, alpha=0.75,
			Bitmap { x=0,y=0,w=2*ui.itemWidth,h=2*ui.itemHeight, image="item/"..filled[n].name.."_big" },
			TightText { x=2*ui.itemWidth+2,y=y,h=2*ui.itemHeight, label=text, font=popupFont, flags=kHAlignLeft+kVAlignTop },
		}
	else
		local y = 2 * bsgFontHeight(popupFont)
		return MakeRollover
		{
			x=0,y=0, color=PopupColor, inset=2, alpha=0.75,
			TightText { x=0,y=0,h=h, label=LabelString("click_ingredient"), font=popupFont, flags=kHAlignLeft+kVAlignTop },
		}
	end
end

local slotButtons = {}
if possibleCount > 0 then
	for i=1,slotCount do
		local angle = 360 * (i-1) / slotCount
		local x = 75 + 50 * bsgSin(angle) - ui.itemWidth / 2
		local y = 75 - 50 * bsgCos(angle) - ui.itemHeight / 2
		
		local n = i
		table.insert(slotButtons,
			Rollover { x=x,y=y, w=ui.itemWidth,h=ui.itemHeight,
				Bitmap { name="slot_"..tostring(n), image="" },
				contents="LabSlotRollover("..tostring(n)..")",
				rolloverSound="lab_rollover",
				command=function() if filled[n] then ClearSlot(n) end end,
			})
	end
end

local ingredientButtons = {}
local xLeft = 4
local x = xLeft
local y = 0

local jarGraphics = { "image/lab_jar_up","image/lab_jar_down","image/lab_jar_over" }
local function AddIngredientsByType(type,x,y,dx,dy)
	dy = dy or 0
	local xLeft = x
	for ing in LItem:IngredientsByType(type) do
--		if ing.inventory > 0 then
		if self.stock[ing.name] then
			local n = ing.name
--[[
			table.insert(ingredientButtons,
				Button { x=x,y=y, graphics=jarGraphics, command=function() AddIngredient(n) end,
					sound="sfx/lab_select.ogg", rolloversound="sfx/lab_rollover.ogg",
						Bitmap { x=10,y=28, image="item/"..n },
				})
]]--
			if possibleCount > 0 then
				-- Rollover with click-to-add
				table.insert(ingredientButtons,
					Rollover { x=x,y=y,w=52,h=75,
						contents=ing:RolloverTarget("click_add"),
						command=function() AddIngredient(n) end,
						rolloverSound="lab_rollover",
						Bitmap { x=0,y=0, image="image/lab_jar_up" },
						Bitmap { x=10,y=28, image="item/"..n },
					})
			else
				-- Rollover without click-to-add
				table.insert(ingredientButtons,
					Rollover { x=x,y=y,w=52,h=75,
						contents=ing:RolloverTarget(),
						Bitmap { x=0,y=0, image="image/lab_jar_up" },
						Bitmap { x=10,y=28, image="item/"..n },
					})
			end
				
			x = x + dx
			if x > 282 then
				y = y +  dy
				x = xLeft
			end
		end
	end
	if finalInc then y = y + 34 end
end

local h = bsgFontHeight(smallDialogFont)

AddIngredientsByType("it_cacao", 0, 10, 41)
AddIngredientsByType("it_basic", 61-3, 73, 43)
AddIngredientsByType("it_nut", 41-4, 137, 43)
AddIngredientsByType("it_fruit", 20-5, 201, 43)
AddIngredientsByType("it_flavor", 0, 265, 41)
AddIngredientsByType("it_secret", 10, 340, 56.75, 64)

local historyIcons = {}
local y = 25
for i=1,5 do
	local x=0
	for j=1,6 do
		table.insert(historyIcons, Bitmap {x=x,y=y, name="history_"..tostring(i).."_"..tostring(j), w=30,h=30, scale=30/ui.itemWidth })
		table.insert(historyIcons, Bitmap {x=x+3,y=y, name="bad_"..tostring(i).."_"..tostring(j), image="image/missed_ingredient",scale=30/43 })
		x = x + 30
	end
	y = y + 35
end

local buttonX = charX + (ui.charW - ui.devButtonW)/2

MakeDialog
{
	name="lab",
	Frame
	{
		x=windowX,y=50+ui.yOffset,w=width,h=height,fit=true,
		Bitmap { image="image/lab_backdrop" },
		
		Text { x=317,y=24,h=kMax,w=163,
			name="reaction", label="", font=dialogFont,
			flags=kHAlignLeft+kVAlignTop },
--			flags=kHAlignCenter+kVAlignTop },

		Group(ingredientButtons),
		
		Window { x=459, y=241, h=210,w=181,
			Text { x=0,y=0, w=kMax,h=20, label="lab_history", font=smallDialogFont},
			Group(historyIcons),
		},

		Group(slotCountButtons),
		Text { x=310,y=285, w=charX-310,h=20, name="lab_inventing", label="#"..GetString("lab_inventing", "", GetString(type.name)) },
--		Text { x=310,y=305, w=charX-166,h=20, label="lab_select", name="lab_select" },
	
		TextButton { x=310+75 - ui.devButtonW*.8/2, y=140+10-ui.devButtonH*.8, scale=.8,
			name="check", label="tasteit", command=Submit,
			sound="sfx/lab_tasteit_click.ogg", rolloversound="sfx/lab_tasteit_rollover.ogg" },

		Bitmap { x=310,y=140, name="tray", image="image/tray"..tostring(slotCount),
			Group(slotButtons) },
		
		TextButton { x=buttonX,y=442, name="help", label="help", graphics=SmallButtonGraphics, command=function() HelpDialog(11) end },
		TextButton { x=buttonX+62,y=442, name="ok", label="close", command=CloseLab, graphics=MediumButtonGraphics },
			
	},
	BuildingBanner(self),
	Window { x=charX+windowX,y=0, w=ui.charW,h=ui.charH, fit=true,
		Bitmap { x=8+11, y=0, image=neutral, name="taster" },
		Bitmap { x=0,y=231, image="image/name_tag",
			Text { x=17,y=15,w=156,h=19, font=charNameFontDark,
				label=LabelString(char.name), flags=kVAlignCenter+kHAlignCenter }, },
	},
}

ClearSlots()
UpdateHistory()

-- Must know another recipe of a given type before you can invent one
local availableTypes = 0
for name,_ in pairs(typeCounts) do
	local t = LProductType:ByName(name)
	if t.knownCount == 0 or (not possibleTypes[name]) then
		EnableWindow("set_"..name, false)
		EnableWindow("lbl_"..name, false)
	else
		availableTypes = availableTypes + 1
	end
end

if availableTypes < 2 then EnableWindow("lab_select", false) end
SetButtonToggleState("set_"..type.name, true)

if possibleCount == 0 then
	local total = 0
	for _,type in pairs(LProductType._ByIndex) do
		total = total + type.knownCount
	end
	if total == 72 then SetLabel("reaction",GetString("lab_complete"))
	else SetLabel("reaction",GetString("lab_norecipes"))
	end
	EnableWindow("tray", false)
	EnableWindow("lab_inventing", false)
else
	SetLabel("reaction",GetString("lab_instructions"))
end
